<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check database connection
if (!$pdo) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit;
}

// Check if user is logged in and has permission
if (!isLoggedIn() || !hasPermission('manage_products')) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

error_log('User ID: ' . ($_SESSION['user_id'] ?? 'none'));
error_log('Permissions: ' . print_r($_SESSION['permissions'] ?? [], true));

header('Content-Type: application/json');

try {
    // Get JSON data
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    
    if (!$data) {
        throw new Exception('Invalid JSON data received');
    }

    // If this is a manual adjustment (no supplier_id and no unit_price)
    if (isset($data['product_id']) && isset($data['quantity']) && isset($data['reason'])) {
        $product_id = intval($data['product_id']);
        $quantity = intval($data['quantity']);
        $serial_numbers = $data['serial_no'] ?? [];
        $category_id = $data['category_id'] ?? null;
        $price = isset($data['price']) ? floatval($data['price']) : null;
        $cost_price = isset($data['cost_price']) ? floatval($data['cost_price']) : null;
        $description = $data['description'] ?? null;
        $adjustment_type = $data['adjustment_type'] ?? null;
        $reason = $data['reason'];
        $receipt_id = isset($data['receipt_id']) ? intval($data['receipt_id']) : null;

        if ($product_id <= 0 || $quantity == 0 || empty($reason)) {
            throw new Exception('Missing or invalid required fields for manual adjustment');
        }

        // Start transaction
        $pdo->beginTransaction();

        // Optionally update product details if provided
        if ($price !== null || $cost_price !== null || $description !== null || $category_id !== null) {
            $updateFields = [];
            $updateParams = [];
            if ($price !== null) { $updateFields[] = 'price = ?'; $updateParams[] = $price; }
            if ($cost_price !== null) { $updateFields[] = 'cost_price = ?'; $updateParams[] = $cost_price; }
            if ($description !== null) { $updateFields[] = 'description = ?'; $updateParams[] = $description; }
            if ($category_id !== null) { $updateFields[] = 'category_id = ?'; $updateParams[] = $category_id; }
            if ($updateFields) {
                $updateParams[] = $product_id;
                $stmt = $pdo->prepare("UPDATE products SET ".implode(', ', $updateFields)." WHERE id = ?");
                $stmt->execute($updateParams);
            }
        }

        // Update product quantity
        $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
        $stmt->execute([$quantity, $product_id]);

        // If a receipt_id is provided, update the supplier_receipts status to 'yes'
        if ($receipt_id) {
            $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = 'yes' WHERE id = ?");
            $stmt->execute([$receipt_id]);
        }

        // Add serial numbers if provided
        $serial_numbers = [];
        if (!empty($serial_numbers)) {
            addProductSerialNumbers($product_id, $serial_numbers, 'manual_adjustment', $pdo->lastInsertId());
        }

        // Log the adjustment
        $stmt = $pdo->prepare("
            INSERT INTO stock_adjustments 
            (product_id, quantity, reason, adjusted_by, adjusted_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $product_id, 
            $quantity, 
            $reason,
            $_SESSION['user_id']
        ]);
        $adjustment_id = $pdo->lastInsertId();

        // Log to system logs
        $product_stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
        $product_stmt->execute([$product_id]);
        $product = $product_stmt->fetch();
        $details = sprintf(
            "Product: %s (ID: %d), Adjustment: %+d, Reason: %s, By: %s (User ID: %d)",
            $product['name'],
            $product_id,
            $quantity,
            $reason,
            $_SESSION['username'],
            $_SESSION['user_id']
        );
        logSystemAction('stock_adjusted', $details);

        $pdo->commit();
        echo json_encode(['success' => true, 'message' => 'Stock adjusted successfully']);
        exit;
    }

    if (!isset($data['product_id']) || !isset($data['supplier_id']) || !isset($data['quantity']) || !isset($data['unit_price'])) {
        throw new Exception('Missing required fields');
    }
    
    $product_id = intval($data['product_id']);
    $supplier_id = intval($data['supplier_id']);
    $quantity = intval($data['quantity']);
    $unit_price = floatval($data['unit_price']);
    $serial_numbers = $data['serial_numbers'] ?? [];
    
    // Validate input
    if ($product_id <= 0 || $supplier_id <= 0 || $quantity <= 0 || $unit_price <= 0) {
        throw new Exception('Invalid input values');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    // Check if product exists
    $stmt = $pdo->prepare("SELECT id, name FROM products WHERE id = ?");
    $stmt->execute([$product_id]);
    $product = $stmt->fetch();
    
    if (!$product) {
        throw new Exception('Product not found');
    }
    
    // Find the existing approved supplier receipt
    $stmt = $pdo->prepare("SELECT id FROM supplier_receipts WHERE supplier_id = ? AND product_id = ? AND status = 'approved' LIMIT 1");
    $stmt->execute([$supplier_id, $product_id]);
    $receipt = $stmt->fetch();

    if (!$receipt) {
        throw new Exception('No approved supplier receipt found for this product.');
    }
    $receipt_id = $receipt['id'];

    // Update the status to completed
    $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = 'completed' WHERE id = ?");
    $stmt->execute([$receipt_id]);

    // Update product quantity
    $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
    $stmt->execute([$quantity, $product_id]);
    
    // Log the adjustment
    $stmt = $pdo->prepare("
        INSERT INTO stock_adjustments 
        (product_id, quantity, reason, adjusted_by, adjusted_at, serial_numbers) 
        VALUES (?, ?, ?, ?, NOW(), ?)
    ");
    $stmt->execute([
        $product_id, 
        $quantity, 
        'Added from supplier receipt #' . $receipt_id,
        $_SESSION['user_id'],
        json_encode($serial_numbers)
    ]);
    
    // Log to system logs
    $details = sprintf(
        "Product: %s (ID: %d), Added: %d from supplier receipt #%d, Serial Numbers: %s, By: %s (User ID: %d)",
        $product['name'],
        $product_id,
        $quantity,
        $receipt_id,
        implode(', ', $serial_numbers),
        $_SESSION['username'],
        $_SESSION['user_id']
    );
    logSystemAction('stock_added_from_supplier', $details);
    
    // Commit transaction
    $pdo->commit();
    echo json_encode(['success' => true, 'message' => 'Stock added successfully']);
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log('Error in add_stock.php: ' . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} 